package util;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JacksonException;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * @Desc jackson util
 * @Author Square
 * @Org www.legend.tech
 * @Date 2023/5/15 16:06
 **/
public class JacksonUtil {

    private JacksonUtil() {
    }

    /**
     * It converts a JSON string to a Java object.
     *
     * @param jsonStr The string to be parsed
     * @param clazz   The class of the object to be converted.
     * @return An object of type T
     */
    public static <T> T parse(String jsonStr, Class<T> clazz) {
        try {
            return getObjectMapper().readValue(jsonStr, clazz);
        } catch (JacksonException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * It converts an object to a JSON string.
     *
     * @param o The object to be converted to json
     * @return A JSON string representation of the object.
     */
    public static String toJsonString(Object o) {
        try {
            return getObjectMapper().writeValueAsString(o);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * This function converts an object to a JSON string and excludes any null values.
     *
     * @param o The parameter "o" is an object that will be converted to a JSON string with null values excluded. The
     *          specific type of object is not specified in this method and can be any Java object.
     * @return The method is returning a JSON string representation of the input object with null values excluded.
     */
    public static String toJsonStringNoNull(Object o) {
        try {
            ObjectMapper objectMapper = getObjectMapper();
            objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
            return objectMapper.writeValueAsString(o);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * get ObjectMapper
     *
     * @return A new ObjectMapper object.
     */
    private static ObjectMapper getObjectMapper() {
        ObjectMapper mapper = new ObjectMapper();
        mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        return mapper;
    }

    /**
     * It converts a JSON string to a Java object.
     *
     * @param json          The json string to be parsed
     * @param typeReference The type of the object to be converted.
     */
    public static <T> T parse(String json, TypeReference<T> typeReference) {
        try {
            return getObjectMapper().readValue(json, typeReference);
        } catch (JacksonException e) {
            throw new RuntimeException(e);
        }
    }

}
