package util;

import lombok.Data;
import org.apache.commons.codec.binary.Base64;

import java.lang.reflect.Field;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.*;
import java.util.function.Function;

/**
 * @Desc api sign util
 * @Author Square
 * @Org www.legend.tech
 * @Date 2023/5/15 11:06
 **/
@Data
public class ApiSignUtils {

    /**
     * @param obj
     * @return
     * @throws IllegalAccessException
     */
    public static Map<String, Object> objectToMap(Object obj) throws IllegalAccessException {
        Map<String, Object> map = new HashMap<String, Object>();
        Class<?> clazz = obj.getClass();
        for (Field field : clazz.getDeclaredFields()) {
            field.setAccessible(true);
            String fieldName = field.getName();
            Object value = field.get(obj);
            map.put(fieldName, value);
        }
        return map;
    }

    /**
     * @param param
     * @return
     */
    public static String mapToString(Map param) {
        Set set = param.keySet();
        TreeSet<String> nameSet = new TreeSet<>(set);
        StringBuilder builder = new StringBuilder();
        int i = 0;
        for (String name : nameSet) {
            Object value = param.get(name);
            if (value == null) {
                continue;
            }
            if (value instanceof Collection) {
                value = listToStr(((Collection) value));
            }
            if (value instanceof Map) {
                value = listToStr(((Map) value).values());
            }
            builder.append(name).append("=").append(value);
            i++;
            if (i < nameSet.size()) {
                builder.append("&");
            }
        }
        String res = builder.toString();
        if (res.endsWith("&")) {
            return res.substring(0, res.length() - 1);
        } else {
            return res;
        }
    }

    /**
     * @param value
     * @return
     */
    private static Object listToStr(Collection value) {
        Optional reduce = value.stream().map(new Function<Object, String>() {
            @Override
            public String apply(Object b) {
                String c = null;
                if (b instanceof Collection) {
                    c = listToStr(((Collection) b)).toString();
                } else if (b instanceof Map) {
                    c = listToStr(((Map) b).values()).toString();
                } else {
                    c = b.toString();
                }
                return c;
            }
        }).reduce((a, b) -> a.toString() + b.toString());
        return reduce.get();
    }

    /**
     * 获取公钥
     *
     * @param publicKey 公钥字符串
     * @return
     */
    public static PublicKey getPublicKey(String publicKey) throws Exception {
        KeyFactory keyFactory = KeyFactory.getInstance("RSA");
        byte[] decodedKey = Base64.decodeBase64(publicKey.getBytes());
        X509EncodedKeySpec keySpec = new X509EncodedKeySpec(decodedKey);
        return keyFactory.generatePublic(keySpec);
    }

    /**
     * 获取私钥
     *
     * @param privateKey 私钥字符串
     * @return
     */
    public static PrivateKey getPrivateKey(String privateKey) throws Exception {
        KeyFactory keyFactory = KeyFactory.getInstance("RSA");
        byte[] decodedKey = Base64.decodeBase64(privateKey.getBytes());
        PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(decodedKey);
        return keyFactory.generatePrivate(keySpec);
    }


    /**
     * 验签
     *
     * @param data            原始对象
     * @param publicKeyString 公钥字符串
     * @return 是否验签通过
     */
    public static boolean verify(Object data, String publicKeyString) {
        try {
            Map<String, Object> objectMap = null;
            if (data instanceof Map) {
                objectMap = (Map<String, Object>) data;
            } else {
                objectMap = objectToMap(data);
            }
            String sign = (String) objectMap.get("sign");
            objectMap.remove("sign");
            String srcData = mapToString(objectMap);

            PublicKey publicKey = getPublicKey(publicKeyString);
            byte[] keyBytes = publicKey.getEncoded();
            X509EncodedKeySpec keySpec = new X509EncodedKeySpec(keyBytes);
            KeyFactory keyFactory = KeyFactory.getInstance("RSA");
            PublicKey key = keyFactory.generatePublic(keySpec);
            Signature signature = Signature.getInstance("SHA256WithRSA");
            signature.initVerify(key);
            signature.update(srcData.getBytes());
            return signature.verify(Base64.decodeBase64(sign.getBytes()));
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    public static boolean verify(String sign, Map<String, String> prams, String publicKeyString) {
        String srcData = mapToString(prams);
        return verify(sign, srcData, publicKeyString);
    }

    public static boolean verify(String sign, String srcData, String publicKeyString) {
        try {
            PublicKey publicKey = getPublicKey(publicKeyString);
            byte[] keyBytes = publicKey.getEncoded();
            X509EncodedKeySpec keySpec = new X509EncodedKeySpec(keyBytes);
            KeyFactory keyFactory = KeyFactory.getInstance("RSA");
            PublicKey key = keyFactory.generatePublic(keySpec);
            Signature signature = Signature.getInstance("SHA256WithRSA");
            signature.initVerify(key);
            signature.update(srcData.getBytes());
            return signature.verify(Base64.decodeBase64(sign.getBytes()));
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    /**
     * 加签
     *
     * @param data             原始对象
     * @param privateKeyString 私钥字符串
     * @return 加签名
     */
    public static String sign(Map<String, Object> data, String privateKeyString) {
        return sign(mapToString(data), privateKeyString);
    }

    /**
     * 加签
     *
     * @param privateKeyString 私钥字符串
     * @return 加签名
     */
    public static String sign(String srcData, String privateKeyString) {
        try {
            PrivateKey privateKey = getPrivateKey(privateKeyString);
            byte[] keyBytes = null;
            PKCS8EncodedKeySpec keySpec = null;
            KeyFactory keyFactory = null;
            PrivateKey key = null;
            Signature signature = null;
            String str = null;
            keyBytes = privateKey.getEncoded();
            keySpec = new PKCS8EncodedKeySpec(keyBytes);
            keyFactory = KeyFactory.getInstance("RSA");
            key = keyFactory.generatePrivate(keySpec);
            signature = Signature.getInstance("SHA256withRSA");
            signature.initSign(key);
            signature.update(srcData.getBytes());
            str = new String(Base64.encodeBase64(signature.sign()));
            return str;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    static class BB {
        public String sign;
        public String username;
    }

    public static void main(String[] args) {
//        String publicKey = "MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQCFp/o6Fgptc1+m2wWLIX76EYUAc3Q/qEKxd+LwIWUAEJA2052cSloV+73w/2RFwBGQfVgqwTgB0I2dIqZKk6Ga2fke7HjuFCcwE6UH2rAJCL426FN+47Woe7b1/BluhEdxK11W1ExBiXT1BPrEYxjEyi62NoZ7dd/aXskWW/4owQIDAQAB";
//        String privateKey = "MIICdwIBADANBgkqhkiG9w0BAQEFAASCAmEwggJdAgEAAoGBAIWn+joWCm1zX6bbBYshfvoRhQBzdD+oQrF34vAhZQAQkDbTnZxKWhX7vfD/ZEXAEZB9WCrBOAHQjZ0ipkqToZrZ+R7seO4UJzATpQfasAkIvjboU37jtah7tvX8GW6ER3ErXVbUTEGJdPUE+sRjGMTKLrY2hnt139peyRZb/ijBAgMBAAECgYB+X5ta/KgTYPlZ/NE7AgrbS2FkUcXiwqd8wYxT3F9xNjgFfwURV8Ph8Utg6TnYMYjqA3ovws2/+X9MygxNi1VKhTk0wmqGYOh/A9bcMgwTxwiKYrRHa4fCsIeTHvTUlJduPbPFNdNoKanfkUmEY/RKuXNpjwwCMBKC8lPU8Yf4AQJBAL93aMGilyBKwWgARYTu3QjDzRvsbPGLJ78z8tFm+It0K3FZblQZHkkInRfe//94u07FQYD7CEXEkpOOlGw++2ECQQCytG+MachK/qogcbRTfU/Ywzi2pxiwZBFF06YDGD3MBoN348EygQd8IwhaRPc9Ro7/mKwmNxvJLTCF3n1z2olhAkEAthLliY4GTO0J8s6fTYSgqR8Zw74W87Oy/n52IG/dVzAHx+iVpF+mjykjynmD0/uJk6S9X9xYx4/hG3Tw6+cQwQJAZnCIgTUiW9sgKvkDJdjvrJI8MHgY7wRt4lkxERUUV6BkPJWSaiEM6VwrrZNbJwULc/lQtZdhljERX6vqvj7IAQJBAKwmkRdK0k80611x2ETBCqEdncWnupHrXQhX3WTmE8qJP5e+anO9VbLNRbytD2jQatkLi1CeJYpiGdeCupKhfCw=";
//        Map<String,Object> params = new HashMap<>();
//        params.put("username","vliu");
//        String sign = sign(params,privateKey);
//        System.out.println("生成签名:"+sign);
//        BB bb = new BB();
//        bb.sign = sign;
//        bb.username = "vliu";
//        System.out.println(verify(bb,publicKey));


        long timestamp = System.currentTimeMillis() / 1000;

        long Nonce = System.currentTimeMillis() / 10000;


        String publicKey = "MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAy4zuU6NKi+48hClEsneBOjp9G8XnXNpT76mEw3L0yzuhXaa6hgPPIL00P0ZQ//EZDHr2YxPDm8H6mYlRcAya6nBtRdJNU76OgULTNMrkSnMG925XIbtIzICWvYrNnSW+kTfh5zYgEki3iDr9ChhQWJpX3b1ky3vd/QqsJ7JJdpLUKbMLYXUN4aRjH3WVyeh4Pm/MO1g3cu8r+Gz7uxdU1/bJs0ZxgENDm7K4btsy0kLcmLpcfgDUddMSr9Ejv8d6jVwUpJVKl/aa1QdrGKWODrqLPPdSxTrMdi8nImlT0pU3pJEFmr55Z3pnkABivgP6IlMh4WrSDQDG6wVa/xPiQQIDAQAB";
        Map<String, Object> params = new HashMap<>();
        params.put("merchant_ip", "127.0.0.1");
        params.put("remark", "信用卡API商户");
        params.put("business_type", "HyperCard");
        params.put("timestamp", timestamp);
        params.put("Nonce", Nonce);

        String sign = sign(params, publicKey);
        System.out.println("生成签名:" + sign);
//        BB bb = new BB();
//        bb.sign = sign;
//        bb.username = "vliu";
//        System.out.println(verify(bb,publicKey));

    }
}
